/*  gnoesyCaliCaroNhsqc.c

    3D C13,N15 edited noesy with separation via the carbon of the destination site
         recorded on a water sample 
    includes optional "WET" water suppression in relaxation delay
    includes optional magic-angle gradients

    Uses three channels:
         1)  1H             - carrier @ water  
         2) 13C             (CA/CB)
         3) 15N             

    Set dm = 'nnny', [13C decoupling during acquisition].
    (13C decoupling in t1 is done by a composite pulse)
    Set dm2 = 'nyny', [15N dec during t1 and acquisition] 

    Must set phase = 1,2 and phase2 = 1,2 for States-TPPI
    acquisition in t1 [H]  and t2 [C].

    Set f1180 = 'y' and f2180 = 'y' for (90, -180) in F1 and (90, -180) in  F2.    


    Modified by L. E. Kay to allow for simult N, C acq   July 19, 1993
    original code: noesyc_pfg_h2o_NC_dps.c
    Modified for dps and magic angle gradients D.Mattiello, Varian, 6/8/94
    Modified for vnmr5.2 (new power statements, use status control in t1)
      GG. Palo Alto  16jan96
    Modified to use only z-gradients 
    Modified to use pwC/pwClvl/pwN/pwNlvl/compC as in other BioPack/
     sequences. The parameter "compN" is added to permit accurate lower power
     N15 pulses. This parameter should be calibrated by determining the pw90
     for N15 at the normal pwNlvl(pw1) and 3db lower(pw2). Then,      
     compN=pw2/(2.0*pw1). Use ghn_co or gNhsqc for these calibrations.

    Modified to use BioPack-style of C=O decoupling
     
    STUD DECOUPLING.   SET STUD='y':
       Setting the flag STUD='y' overrides the decoupling parameters listed in
       dg2 and applies STUD+ decoupling instead.  In consequence is is easy
       to swap between the decoupling scheme you have been using to STUD+ for
       an easy comparison.  The STUD+ waveforms are calculated for your field
       strength at the time of BioPack installation and RF power is 
       calculated within the pulse sequence.  The calculations are for the most 
       peaks being greater than 90% of ideal. If you wish to compare different 
       decoupling schemes, the power level used for STUD+ can be obtained from 
       dps - subtract 3dB from that level to compare to decoupling schemes at
       a continuous RF level such as GARP.  The value of 90% has
       been set to limit sample heating as much as possible.  If you wish to 
       change STUD parameters, for example to increase the quality of decoupling
       (and RF heating) change the 95% level for the centerband
       by changing the relevant number in the macro makeSTUDpp and 
       rerun the macro (don't use 100% !!).  (At the time of writing STUD has
       been coded to use the coarse attenuator, because the fine attenuator
       is not presently included in the fail-safe calculation of decoupling 
       power which aborts an experiment if the power is too high - this may
       lower STUD efficiency a little).

       Ref: Pascal et.al.,J.Magn.Reson. B103,197-201(1994)
       GG, palo alto, 7 july 1998

   Modified by A. Eletski, Aug 2007, based on cnnoesy 

   * sw2N is the spectral width for 15N. Should be different from sw2 (13C)
   * 13C/15N t1 initial evolution is half-dwell only;
     f2180 flag is checked to be 'y' for consistency
     Initial evolution is set to almost zero evolution if ni2=0
   * uses adiabatic 13C inversion pulses, set C180='stC140' or C180='stC200'
   * adiabatic pulse width pwC180 is set to 1 ms; composite 13C inversion pulses are disabled 
   * simultaneous hard pulses are avoided, peak power can be used on 13C and 15N channels 
   * Indirect evolution modified to get correct first two points
     Backward linear prediction not needed
   * 15N 180 degree pulses changed to 90-180-90 for broadband preformance
   * Added jnh parameter for J_NH coupling constant
   * INEPT delays are determined by jnh; jch is used to shift the 13C 180 degree pulses
   * Added a flipN flag. Set flipN='y' to change the sign of 15N peaks
   * Optimized water suppression; fixed phase cycling.

   
   Usage: set dof to 35-40 ppm or 65-70 ppm, dof2 to 120 ppm
          set sw2 to 25-30 ppm, sw2N to 30 ppm

          To avoid overlap of amide and aromatic strips record 2D H-C/N planes
          and optimize these settings.

          In case of excessive overlap of 15N and 13Caro peaks you can acquire two
          separate spectra with flipN='n' and flipN='y'. Add and subtract them to 
          extract individual 15N and 13C parts.

          rof2 and alfa values leading to zero linear phase (lp=0) are different
          from those of most BioPack pulse sequences. Record a 2D plane first, and
          use calfa macro to determine correct rof2 and alfa.

          Use wurst140 decoupling for 13C.

          For best results set nt=4 as a minimum. The first two scans suppress
          axial peaks in the 15N/13C dimension - needed since 13C is often folded.
          The first four scans also suppress axial peaks in the 1H NOE dimension.
          Maximum water suppression is achieved in eight scans.

*/

#include <standard.h>

static int  phi1[4]  = {0,0,2,2},
            phi2[2]  = {0,2},
            phi3[4]  = {0,0,2,2},
            rec[8]   = {0,2,2,0,2,0,0,2},
            phi5[8]  = {0,0,0,0,2,2,2,2},
            phi6[1]  = {0};
                    
static double d2_init=0.0, d3_init=0.0;

pulsesequence()
{
/* DECLARE VARIABLES */

 char     
            f1180[MAXSTR],    /* Flag to start t1 @ halfdwell             */
            f2180[MAXSTR],    /* Flag to start t2 @ halfdwell             */
            mag_flg[MAXSTR],  /* magic-angle gradients                    */
            C180[MAXSTR],     /* adiabatic 13C inversion pulse            */
            wet[MAXSTR],      /* Flag to select optional WET water suppression */
            STUD[MAXSTR],     /* Flag to select adiabatic decoupling      */
            dmm2[MAXSTR],
            msg[MAXSTR],
            flipN[MAXSTR],    /* Flip the phase of 15N peaks by 180 deg */
            stCdec[MAXSTR];   /* contains name of adiabatic decoupling shape */

 int         
             ni2 = getval("ni2"),
             t1_counter,   /* used for states tppi in t1           */ 
             t2_counter;   /* used for states tppi in t2           */ 

 double     stdmf = getval("dmf80"),     /* dmf for 80 ppm of STUD decoupling */
            rf80 = getval("rf80"),       	  /* rf in Hz for 80ppm STUD+ */
/*            pwC180=getval("pwC180"),     /* adiabatic 13C pw180 */
            pwC180= 1.0e-3,
            rfst,                        /* fine power level for adiabatic pulse*/
            studlvl,	  /* coarse power for STUD+ decoupling */
            pwC10 = getval("pwC10"), 
                        /* 180 degree selective sinc pulse on CO(174ppm) */
/*            rf7, */
  	                /* fine power for the pwC10 ("offC10") pulse */
            rf0,                        /* full fine power */
            compC = getval("compC"), 
                        /* adjustment for C13 amplifier compression */
             tau1,         /*  t1 delay */
             tau2,         /*  t2 delay */
             tau2N,         /*  t2 delay (N) */
             corr,         /*  correction for t2  */
             jch,          /*  CH coupling constant */
             jnh,          /*  CH coupling constant */
             pwN,          /* PW90 for 15N pulse              */
             pwC,          /* PW90 for c nucleus @ pwClvl         */
             pwClvl,        /* power level for 13C pulses on dec1  */
             pwNlvl,       /* high dec2 pwr for 15N hard pulses    */
             mix,          /* noesy mix time                       */
             sw1,          /* spectral width in t1 (H)             */
             sw2,          /* spectral width in t2 (C) (3D only)   */
             sw2N,         /* spectral width in t2 (N) (3D only)   */
             gtw = getval("gtw"), gswet=getval("gswet"), wetpw=getval("wetpw"),
             dz = getval("dz"), gstab=getval("gstab"),
             maxtime,
             gt0,
             gt1,
             gt2,
             gzcal,
             gt3,
             gt4,
             gt5,
             gt6,
             gt7,
             gzlvl0, 
             gzlvl1, 
             gzlvl2, 
             gzlvl3, 
             gzlvl4, 
             gzlvl5,
             gzlvl6,
             gzlvl7; 


/* LOAD VARIABLES */
  getstr("wet",wet);
  getstr("mag_flg",mag_flg);
  getstr("C180",C180);
  getstr("f1180",f1180);
  getstr("f2180",f2180);
  getstr("STUD",STUD);
  getstr("dmm2",dmm2);
  getstr("flipN",flipN);


    mix  = getval("mix");
    sw1  = getval("sw1");
    sw2  = getval("sw2");
    sw2N = getval("sw2N");
  jch = getval("jch"); 
  jnh = getval("jnh");
  pwC = getval("pwC");
  pwN = getval("pwN");
  pwClvl = getval("pwClvl");
  pwNlvl = getval("pwNlvl");
  gzcal = getval("gzcal");
  gt0 = getval("gt0");
  gt1 = getval("gt1");
  gt2 = getval("gt2");
  gt3 = getval("gt3");
  gt4 = getval("gt4");
  gt5 = getval("gt5");
  gt6 = getval("gt6");
  gt7 = getval("gt7");
  gzlvl0 = getval("gzlvl0");
  gzlvl1 = getval("gzlvl1");
  gzlvl2 = getval("gzlvl2");
  gzlvl3 = getval("gzlvl3");
  gzlvl4 = getval("gzlvl4");
  gzlvl5 = getval("gzlvl5");
  gzlvl6 = getval("gzlvl6");
  gzlvl7 = getval("gzlvl7");

/* LOAD PHASE TABLE */
  settable(t1,4,phi1);
  settable(t2,2,phi2);
  settable(t3,4,phi3);
  settable(t4,8,rec);
  settable(t5,8,phi5);
  settable(t6,1,phi6);

/* CHECK VALIDITY OF PARAMETER RANGES */

    if((dm[A] == 'y') || (dm[B] == 'y') || (dm[C] == 'y' ))
    {
        text_error("incorrect dec1 decoupler flags! Should be 'nnny' 13C is decoupled in t1 by pulse ");
        psg_abort(1);
    }

    if((dm2[A] == 'y' || dm2[C] == 'y' ))
    {
        text_error("incorrect dec2 decoupler flags! Should be 'nyny' ");
        psg_abort(1);
    }

    if( dpwr > 52 )
    {
        text_error("don't fry the probe, DPWR too large!  ");
        psg_abort(1);
    }

    if( dpwr2 > 50 )
    {
        text_error("don't fry the probe, DPWR2 too large!  ");
        psg_abort(1);
    }

    if( pw > 200.0e-6 )
    {
        text_error("dont fry the probe, pw too long. Check value ! ");
        psg_abort(1);
    } 


    if( pwN > 200.0e-6 )
    {
        text_error("dont fry the probe, pwN too long. Check value ! ");
        psg_abort(1);
    } 

/*    if( pwN < 25.0e-6 )
    {
        text_error("Use lower power level. Use pwNlvl for pwN>40usec");
        psg_abort(1);
    }
    if( pwC < 12.0e-6 )
    {
        text_error("Use lower power level. Use pwClvl for pwC>12usec");
        psg_abort(1);
    } 
*/
    if( pwC > 200.0e-6 )
    {
        text_error("dont fry the probe, pwC too long. Check value ! ");
        psg_abort(1);
    } 

    if( gt0 > 15e-3 || gt1 > 15e-3 || gt2 > 15e-3 || gt3 > 15e-3 || gt4 > 15e-3 || gt5 > 15e-3 || gt6 > 15e-3 || gt7 > 15e-3 ) 
    {
        text_error("gti values < 15e-3\n");
        psg_abort(1);
    } 

   if( gzlvl3*gzlvl4 > 0.0 ) 
    {
    text_error("gt3 and gt4 must be of opposite sign for optimal water suppression\n");
        psg_abort(1);
     }

   if( f2180[A] != 'y' )
    {
    text_error("f2180 must be set to 'y'\n");
        psg_abort(1);
     }
                                                                                
   if( sw2 < sw2N )
    {    text_error("sw2 must be greater than sw2N!\n", 0.25/(2.0/PI*pwC+pw));
        psg_abort(1);
     }

   if( 0.25/sw2 - (2.0/PI)*pwC - pw < 0 )
    {     sprintf(msg, "sw2 too big! Please set sw2 to %.0f Hz or smaller\n", 0.25/(2.0/PI*pwC+pw));
    text_error(msg);
        psg_abort(1);
     }
                                                                                
   if( 0.25/sw2N - (2.0/PI)*pwC - 0.25/sw2 - (1.0-2.0/PI)*pwC + pw - 2.0e-6 < 0 )
    {
    sprintf(msg, "sw2N too big or sw2 too small! Please set sw2N to < %.0f Hz or set sw2N to > %.0f Hz\n", 0.25/(2.0/PI*pwC+0.25/sw2+(1-2/PI)*pwC-pw+2.0e-6), 0.25/(0.25/sw2N-2/PI*pwN-(1-2/PI)*pwC+pw-2.0e-6) );
    text_error(msg);
        psg_abort(1);
     }

   if ( (strcmp(C180,"stC140") != 0) && (strcmp(C180,"stC200") != 0) )
    {
    text_error("Set C180 to 'stC140' or 'stC200'\n");
        psg_abort(1);
     }

   maxtime = gt2 + pwC180/2.0 + 1.0e-6;
   if (maxtime < gt5 + pwC180/2.0 + 10.0e-6 + 2.0*pwN + 2.0*pwC)
      maxtime = gt5 + pwC180/2.0 + 10.0e-6 + 2.0*pwN + 2.0*pwC;
   if (1/(4.0*jch) < maxtime )
    {
     sprintf(msg, "jch too big! Please set jch to %.0f Hz or less\n", 0.25/maxtime );
     text_error(msg);
     psg_abort(1);
     }   

   maxtime = gt2 + pwN*2.0 + 4.0e-6;
   if (maxtime < gt5 + pwN*2.0 + 2.0e-6)
      maxtime = gt5 + pwN*2.0 + 2.0e-6;
   if (1/(4.0*jnh) < maxtime )
    {
     sprintf(msg, "jnh too big! Please set jnh to %.0f Hz or less\n", 0.25/maxtime );
     text_error(msg);
     psg_abort(1);
     }

   if (1/(4.0*jnh) < 1/(4.0*jch) + pwC180/2.0 + pwN*2 + 2.0e-6)
    {
     sprintf(msg, "jnh too big or jch too small! Please set jnh to < %.0f Hz or jch to > %.0f Hz\n", 0.25/(1/(4.0*jch) + pwC180/2.0 + pwN*2.0 + 2.0e-6), 0.25/(1/(4.0*jnh) - pwC180/2.0 - pwN*2.0 - 2.0e-6) );
     text_error(msg);
     psg_abort(1);
     }


/*  Phase incrementation for hypercomplex 2D data */

    if (phase1 == 2)
      tsadd(t1,1,4);
    if (phase2 == 2)
      tsadd(t2,1,4);


/* Calculate modifications to phases for States-TPPI acquisition          */

   if( ix == 1) d2_init = d2 ;
   t1_counter = (int) ( (d2-d2_init)*sw1 + 0.5 );
   if(t1_counter % 2) {
      tsadd(t1,2,4);
      tsadd(t4,2,4);    
    }

   if( ix == 1) d3_init = d3 ;
   t2_counter = (int) ( (d3-d3_init)*sw2 + 0.5 );
   if(t2_counter % 2) {
      tsadd(t2,2,4);  
      tsadd(t4,2,4);    
    }

/* 180 degree phase flipping for 15N pulses */
   if(flipN[A] == 'y') {
      tsadd(t6,2,4);
    }


   /* 80 ppm STUD+ decoupling */
	strcpy(stCdec, "stCdec80");
	studlvl = pwClvl + 20.0*log10(compC*pwC*4.0*rf80);
	studlvl = (int) (studlvl + 0.5);
    /* 80ppm sech/tanh ("stC80") inversion */
    /*
	rfst = (4095.0*compC*pwC*4000.0*sqrt((12.07*sfrq/600+3.85)/0.35));
	rfst = (int) (rfst + 0.5);
    */

    if (strcmp(C180,"stC140") == 0) {
        rfst = (compC*4095.0*pwC*4000.0*sqrt((21.0*sfrq/600.0+7.0)/0.35));
        rfst = (int) (rfst + 0.5);
        }
    if (strcmp(C180,"stC200") == 0) {
        rfst = (compC*4095.0*pwC*4000.0*sqrt((30.0*sfrq/600.0+7.0)/0.35));
        rfst = (int) (rfst + 0.5);
        }


    if( pwC > (26.0e-6*600.0/sfrq) )
	{ text_error("Increase pwClvl so that pwC < 26*600/sfrq");
	  psg_abort(1); }

    /* maximum fine power for pwC pulses */
	rf0 = 4095.0;


    /* 180 degree one-lobe sinc pulse on CO, null at Ca 139ppm away */
    /* 13C offset (dof) should be at 35ppm to properly decouple C=O */
    /*  (don't need compC in calculation since it has been used above */

/*	rf7 = (4095.0*compC*pwC*2.0*1.65)/pwC10;	/* needs 1.65 times more     */
/*	rf7 = (int) (rf7 + 0.5);		/* power than a square pulse */


    tau1 = d2;
    if (f1180[A] == 'y')
        tau1 += 1.0 / (2.0*sw1);
    tau1 = tau1/2.0;
    if (tau1 < 0.2e-6) tau1 = 0.0;


    /*  f2180 effect hardwired - A.E. */
    tau2 = d3;
    tau2 += ( 1.0 / (2.0*sw2));
    tau2 = tau2/2.0;

   /*  Set up f2180  tau2N = t2   */
    tau2N = d3*sw2/sw2N ;
    tau2N += ( 1.0 / (2.0*sw2N));    /* plus half point delay */
    tau2N = tau2N /2.0;


/* BEGIN ACTUAL PULSE SEQUENCE */


status(A);
   decpower(pwClvl);  /* Set Dec1 power for hard 13C pulses         */
   obspower(tpwr);           /* Set transmitter power for hard 1H pulses */
   dec2power(dpwr2);       /* Set Dec2 to low power for decoupling */
   delay(d1);
   if (wet[A] == 'y') 
    {
     Wet4(zero,one);             /* WET solvent suppression */
     delay(1.0e-4);
    }
   rcvroff();
status(B);
   decoffset(-22*dfrq);              /* set 13C carrier to 70 ppm */
   rgpulse(pw,t1,1.0e-4,2.0e-6);                  /* 90 deg 1H pulse */


   if (( ni > 1 ) && (tau1 > 0.0))
      {
      if (tau1 - 2/PI*pw - 2.0*pwC - 2.0e-6 > 0.0)
         {
         delay(tau1 - 2/PI*pw - 2.0*pwC - 4.0e-6);
         decrgpulse(pwC,zero,0.0,0.0);            /* composite 180 on 13C */
         decrgpulse(2.0*pwC,one,2.0e-6,0.0);
         decrgpulse(pwC,zero,2.0e-6,0.0);
         delay(tau1 - 2/PI*pw - 2.0*pwC - 4.0e-6);
         }
      else
         {
         delay(2.0*tau1 - 4/PI*pw);
         }
      }
   else
      {
      rgpulse(2.0*pw,zero,2.0e-6,2.0e-6);
      }


status(C);
   if (( ni > 1 ) && (tau1 > 0.0))
      rgpulse(pw,zero,2.0e-6,0.0);             /*  2nd 1H 90 pulse   */
   else
      rgpulse(pw,two,2.0e-6,0.0);

/*   rgpulse(pw,zero,2.0e-6,0.0);             /*  2nd 1H 90 pulse   */
   dec2power(pwNlvl);
   if (wet[C] == 'y')
    {
     delay(mix - 4.0*(gtw +wetpw +gswet) -dz -pwC);                    /*  mixing time     */
     decrgpulse(pwC,zero,2.0e-6,2.0e-6); 
     Wet4(zero,one);                          /* WET solvent suppression */
    }
   else
    {
     delay(mix - 10.0e-3);                    /*  mixing time     */
     zgradpulse(gzlvl0, gt0);
     decrgpulse(pwC,zero,2.0e-6,2.0e-6); 
     zgradpulse(gzlvl1, gt1);
     delay(10.0e-3 - gt1 - gt0 - 4.0e-6 - pwC);
    }
   rgpulse(pw,zero,0.0,2.0e-6);
   zgradpulse(gzlvl2, gt2);       /* g3 in paper   */

     decpwrf(rfst);
     delay(1/(4.0*jnh) - gt2 - pwN*2.0 - 4.0e-6);
     dec2rgpulse(pwN,zero,0.0,2.0e-6);
     sim3pulse(2*pw,0.0,2.0*pwN,zero,zero,one,0.0,2.0e-6);
     dec2rgpulse(pwN,zero,0.0,0.0);
     delay(1/(4.0*jnh) - 1/(4.0*jch) - pwC180/2.0 - pwN*2.0 - 2.0e-6);
     decshaped_pulse(C180,pwC180,zero,0.0,2.0e-6);
     zgradpulse(gzlvl2, gt2);
     decpwrf(4095.0);
     delay(1/(4.0*jch) - gt2 - pwC180/2.0 - 1.0e-6);


   rgpulse(pw,one,1.0e-6,2.0e-6);
                if (mag_flg[A] == 'y')
                {
                   magradpulse(gzcal*gzlvl3, gt3);
                }
                else
                {
                   zgradpulse(gzlvl3, gt3);
                }
   delay(gstab);

   decoffset(dof);                   /* set 13C carrier to dof */
   if (ni2 == 0) {
      dec2rgpulse(pwN,t2,0.0,0.0);
      decrgpulse(pwC,t2,2.0e-6,0.0);
      decrgpulse(pwC,zero,0.0,2.0e-6);
      dec2rgpulse(pwN,zero,0.0,0.0);
   }
   else {
      dec2phase(t2);
      dec2rgpulse(pwN,t2,0.0,0.0);
      delay(tau2N - (2.0/PI)*pwN - tau2 - (1.0-2.0/PI)*pwC + pw - 2.0e-6);
      decphase(t2);
      decrgpulse(pwC,t2,2.0e-6,0.0);
      delay(tau2 - pw - (2.0/PI)*pwC );
      rgpulse(2.0*pw,zero,0.0,0.0);
      delay(tau2 - pw - (2.0/PI)*pwC );
      decphase(zero);
      decrgpulse(pwC,zero,0.0,2.0e-6);
      delay(tau2N - (2.0/PI)*pwN - tau2 - (1.0-2.0/PI)*pwC + pw - 2.0e-6);
      dec2phase(zero);
      dec2rgpulse(pwN,t6,0.0,0.0);
   }
      decoffset(-22*dfrq);              /* set 13C carrier to 70 ppm */


                if (mag_flg[A] == 'y')
                {
                   magradpulse(gzcal*gzlvl4, gt4);
                }
                else
                {
                   zgradpulse(gzlvl4, gt4);
                }
   delay(gstab);
   rgpulse(pw,t5,rof1,0.0); 
 
   txphase(zero); decphase(zero); dec2phase(zero);
   zgradpulse(gzlvl5, gt5);

     decpwrf(rfst);
     delay(1/(4.0*jnh) - gt5 - pwN*2.0 - 2.0e-6);
     dec2rgpulse(pwN,zero,0.0,2.0e-6);
     sim3pulse(2*pw,0.0,2.0*pwN,zero,zero,one,0.0,2.0e-6);
     dec2rgpulse(pwN,zero,0.0,0.0);
     delay(1/(4.0*jnh) - 1/(4.0*jch) - pwC180/2.0 - pwN*2 - 2.0e-6);
     decshaped_pulse(C180,pwC180,zero,0.0,2.0e-6);
     zgradpulse(gzlvl5, gt5);
     decpwrf(4095.0);
     delay(1/(4.0*jch) - gt5 - pwC180/2.0 - 10.0e-6 -2.0*pwN - 2.0*pwC);


   decrgpulse(pwC,zero,0.0,0.0);
   decrgpulse(pwC,t3,2.0e-6,0.0);
   dec2rgpulse(pwN,zero,2.0e-6,0.0);
   dec2rgpulse(pwN,t3,2.0e-6,0.0);


/*   rgpulse(pw,t5,2.0e-6,rof2);                        /* flip-back pulse  */
   rgpulse(pw,zero,2.0e-6,rof2);
   rcvron();
   dec2power(dpwr2);
   decpower(dpwr);
   setreceiver(t4);
   if ((STUD[A] == 'y') && (dm[D] == 'y'))
    {
     setstatus(DECch, FALSE, 'c', FALSE, dmf);     /* override status mode */
     decpower(studlvl);
     decprgon(stCdec, 1.0/stdmf, 1.0);
     decon();
     if(dm2[D] == 'y')		
      {
       setstatus(DEC2ch, TRUE, dmm2[D], FALSE, dmf2); 
      }
    }
   else
   status(D);
}                 /* end of pulse sequence */

/* Wet4 - Water Elimination */
Wet4(phaseA,phaseB)
  codeint phaseA,phaseB;
{
  double finepwr,gzlvlw,gtw,gswet,gswet2,wetpwr,wetpw,dz;
  char   wetshape[MAXSTR];
  getstr("wetshape",wetshape);    /* Selective pulse shape (base)  */
  wetpwr=getval("wetpwr");        /* User enters power for 90 deg. */
  wetpw=getval("wetpw");        /* User enters power for 90 deg. */
  dz=getval("dz");
  finepwr=wetpwr-(int)wetpwr;     /* Adjust power to 152 deg. pulse*/
  wetpwr=(double)((int)wetpwr);
  if (finepwr==0.0) {wetpwr=wetpwr+5; finepwr=4095.0; }
  else {wetpwr=wetpwr+6; finepwr=4095.0*(1-((1.0-finepwr)*0.12)); }
  rcvroff();
  obspower(wetpwr);         /* Set to low power level        */
  gzlvlw=getval("gzlvlw");      /* Z-Gradient level              */
  gtw=getval("gtw");            /* Z-Gradient duration           */
  gswet=getval("gswet");        /* Post-gradient stability delay */
  gswet2=getval("gswet2");        /* Post-gradient stability delay */
  Chess(finepwr*0.5059,wetshape,wetpw,phaseA,20.0e-6,rof1,gzlvlw,gtw,gswet);
  Chess(finepwr*0.6298,wetshape,wetpw,phaseB,20.0e-6,rof1,gzlvlw/2.0,gtw,gswet);
  Chess(finepwr*0.4304,wetshape,wetpw,phaseB,20.0e-6,rof1,gzlvlw/4.0,gtw,gswet);
  Chess(finepwr*1.00,wetshape,wetpw,phaseB,20.0e-6,rof1,gzlvlw/8.0,gtw,gswet2);
  obspower(tpwr); obspwrf(tpwrf);     /* Reset to normal power level   */
  rcvron();
  delay(dz);
}
/* Chess - CHEmical Shift Selective Suppression */
Chess(pulsepower,pulseshape,duration,phase,rx1,rx2,gzlvlw,gtw,gswet)  double pulsepower,duration,rx1,rx2,gzlvlw,gtw,gswet;
  codeint phase;
  char* pulseshape;
{
  obspwrf(pulsepower);
  shaped_pulse(pulseshape,duration,phase,rx1,rx2);
  zgradpulse(gzlvlw,gtw);

  delay(gswet);
}
